<?php

namespace Modules\FreescoutInstaller\Services;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

class CatalogService
{
    const INSTANCE_ID_OPTION = 'freescoutinstaller.instance_id';

    public static function fetch()
    {
        $result = [
            'modules' => [],
            'error' => null,
            'raw' => null,
        ];

        try {
            $client = new Client();
            $response = $client->request('GET', 'https://freescout-installer-api.suivi.workers.dev/api/v1/catalog', [
                'timeout' => 8,
                'connect_timeout' => (int) config('app.curl_connect_timeout', 5),
                'proxy' => config('app.proxy'),
            ]);

            $result['raw'] = (string) $response->getBody();
            $status = $response->getStatusCode();

            if ($status < 200 || $status >= 300) {
                $result['error'] = __('Catalog API returned HTTP :status.', ['status' => $status]);
                return $result;
            }

            $payload = json_decode($result['raw'], true);
            if (isset($payload['modules']) && is_array($payload['modules'])) {
                $result['modules'] = $payload['modules'];
            } else {
                $result['error'] = __('Invalid response from catalog API.');
            }
        } catch (GuzzleException $e) {
            $result['error'] = __('Could not load modules catalog. Please try again later.');
        }

        return $result;
    }

    public static function getInstanceId()
    {
        $instanceId = \Option::get(self::INSTANCE_ID_OPTION);
        if (!$instanceId) {
            if (method_exists(\Illuminate\Support\Str::class, 'uuid')) {
                $instanceId = (string) \Illuminate\Support\Str::uuid();
            } else {
                $instanceId = bin2hex(random_bytes(16));
            }
            \Option::set(self::INSTANCE_ID_OPTION, $instanceId);
        }

        return $instanceId;
    }

    public static function getDomain()
    {
        $url = (string) config('app.url');
        $host = parse_url($url, PHP_URL_HOST);
        if (!$host) {
            $host = (string) request()->getHost();
        }
        return strtolower($host);
    }

    public static function fetchTrialStatus($moduleSlug, $domain)
    {
        $result = [
            'status' => null,
            'days_left' => null,
            'end_date' => null,
            'error' => null,
        ];

        try {
            $client = new Client();
            $response = $client->request('GET', 'https://freescout-installer-api.suivi.workers.dev/api/v1/trial/status', [
                'timeout' => 10,
                'connect_timeout' => (int) config('app.curl_connect_timeout', 5),
                'proxy' => config('app.proxy'),
                'query' => [
                    'module_slug' => $moduleSlug,
                    'domain' => $domain,
                ],
            ]);

            $status = $response->getStatusCode();
            if ($status < 200 || $status >= 300) {
                $result['error'] = __('Trial status unavailable (HTTP :status).', ['status' => $status]);
                return $result;
            }

            $payload = json_decode((string) $response->getBody(), true);
            if (is_array($payload)) {
                $result['status'] = $payload['status'] ?? null;
                $result['days_left'] = $payload['days_left'] ?? null;
                $result['end_date'] = $payload['end_date'] ?? null;
            } else {
                $result['error'] = __('Trial status response invalid.');
            }
        } catch (GuzzleException $e) {
            $result['error'] = __('Trial status API unreachable.');
        }

        return $result;
    }

    public static function startTrial($moduleSlug, $domain, $instanceId)
    {
        try {
            $client = new Client();
            $response = $client->request('POST', 'https://freescout-installer-api.suivi.workers.dev/api/v1/trial/start', [
                'timeout' => 10,
                'connect_timeout' => (int) config('app.curl_connect_timeout', 5),
                'proxy' => config('app.proxy'),
                'json' => [
                    'module_slug' => $moduleSlug,
                    'domain' => $domain,
                    'instance_id' => $instanceId,
                ],
            ]);

            $status = $response->getStatusCode();
            if ($status < 200 || $status >= 300) {
                return ['ok' => false, 'error' => __('Trial start failed (HTTP :status).', ['status' => $status])];
            }

            return ['ok' => true, 'error' => null];
        } catch (GuzzleException $e) {
            return ['ok' => false, 'error' => __('Trial start API unreachable.')];
        }
    }

    public static function getInstalledAliases()
    {
        $aliases = [];
        $moduleFiles = glob(base_path('Modules/*/module.json')) ?: [];
        foreach ($moduleFiles as $file) {
            $contents = @file_get_contents($file);
            if ($contents === false) {
                continue;
            }
            $data = json_decode($contents, true);

            // Add the alias from module.json
            if (!empty($data['alias'])) {
                $aliases[] = $data['alias'];
            }

            // Also add the folder name (often matches catalog slug)
            $folderName = basename(dirname($file));
            if ($folderName && !in_array($folderName, $aliases, true)) {
                $aliases[] = $folderName;
            }
        }

        return array_unique($aliases);
    }

    public static function getEnabledAliases()
    {
        $aliases = [];
        if (!class_exists('\\Module')) {
            return $aliases;
        }

        $moduleFiles = glob(base_path('Modules/*/module.json')) ?: [];
        foreach ($moduleFiles as $file) {
            $contents = @file_get_contents($file);
            if ($contents === false) {
                continue;
            }
            $data = json_decode($contents, true);
            $alias = $data['alias'] ?? null;
            $folderName = basename(dirname($file));

            if (!$alias) {
                continue;
            }

            try {
                $module = \Module::findByAlias($alias);
                $isEnabled = false;

                if ($module && method_exists($module, 'isEnabled') && $module->isEnabled()) {
                    $isEnabled = true;
                } elseif (\Module::isEnabled($alias)) {
                    $isEnabled = true;
                }

                if ($isEnabled) {
                    $aliases[] = $alias;
                    // Also add folder name if different
                    if ($folderName && strtolower($folderName) !== strtolower($alias)) {
                        $aliases[] = $folderName;
                    }
                }
            } catch (\Throwable $e) {
                // Ignore module status errors.
            }
        }

        return array_unique($aliases);
    }
}
