@php
    $modules = $modules ?? [];
    $error = $error ?? null;
    $rawInstalledAliases = $installedAliases ?? [];
    $rawEnabledAliases = $enabledAliases ?? [];
    $installedAliases = array_map('strtolower', $rawInstalledAliases);
    $enabledAliases = array_map('strtolower', $rawEnabledAliases);
    $debug = false; // Set to true to enable debug output

    // Normalize function: remove hyphens/underscores for flexible matching
    $normalize = function($str) {
        return strtolower(str_replace(['-', '_'], '', $str));
    };

    // Check if slug matches any alias (flexible matching)
    $isInstalled = function($slug) use ($installedAliases, $normalize) {
        $slugNorm = $normalize($slug);
        foreach ($installedAliases as $alias) {
            if ($alias === strtolower($slug)) return true;
            if ($normalize($alias) === $slugNorm) return true;
            // Check if one contains the other (for freescout-themes vs themes)
            if (strpos($normalize($alias), $slugNorm) !== false) return true;
            if (strpos($slugNorm, $normalize($alias)) !== false) return true;
        }
        return false;
    };

    $isEnabled = function($slug) use ($enabledAliases, $normalize) {
        $slugNorm = $normalize($slug);
        foreach ($enabledAliases as $alias) {
            if ($alias === strtolower($slug)) return true;
            if ($normalize($alias) === $slugNorm) return true;
            if (strpos($normalize($alias), $slugNorm) !== false) return true;
            if (strpos($slugNorm, $normalize($alias)) !== false) return true;
        }
        return false;
    };

    $perPage = 10;
@endphp

@if ($debug)
<div class="alert alert-warning">
    <strong>DEBUG MODE</strong><br>
    <strong>Installed Aliases (raw):</strong> {{ implode(', ', $rawInstalledAliases) ?: 'none' }}<br>
    <strong>Installed Aliases (lowercase):</strong> {{ implode(', ', $installedAliases) ?: 'none' }}<br>
    <strong>Enabled Aliases (raw):</strong> {{ implode(', ', $rawEnabledAliases) ?: 'none' }}<br>
    <strong>Enabled Aliases (lowercase):</strong> {{ implode(', ', $enabledAliases) ?: 'none' }}<br>
    <strong>Module Slugs from Catalog:</strong> {{ implode(', ', array_column($modules, 'slug')) ?: 'none' }}
</div>
@endif

@if (!empty($error))
    <div class="alert alert-danger">
        {{ $error }}
    </div>
@endif

@if (session('success'))
    <div class="alert alert-success">
        {{ session('success') }}
    </div>
@endif
@if (session('warning'))
    <div class="alert alert-warning">
        {{ session('warning') }}
        <a href="{{ url('/modules/list') }}" class="btn btn-xs btn-default" style="margin-left: 10px;">
            {{ __('Enable manually') }}
        </a>
    </div>
@endif

<div class="panel panel-default" style="margin-top: 15px;">
    <div class="panel-heading">
        <div class="row">
            <div class="col-sm-6">
                {{ __('Available Modules') }}
                <span class="badge" id="modules-count">{{ count($modules) }}</span>
            </div>
            <div class="col-sm-6">
                <div class="input-group input-group-sm pull-right" style="max-width: 300px;">
                    <input type="text" class="form-control" id="modules-search" placeholder="{{ __('Search modules...') }}">
                    <span class="input-group-btn">
                        <button class="btn btn-default" type="button" id="modules-search-clear" title="{{ __('Clear') }}">
                            <i class="glyphicon glyphicon-remove"></i>
                        </button>
                    </span>
                </div>
            </div>
        </div>
    </div>
    <div class="panel-body">
        @if (empty($modules))
            <p class="text-muted">{{ __('No modules found.') }}</p>
        @else
            <!-- Filter buttons -->
            <div class="btn-group btn-group-sm margin-bottom" style="margin-bottom: 15px;">
                <button type="button" class="btn btn-default active" data-filter="all">{{ __('All') }}</button>
                <button type="button" class="btn btn-default" data-filter="free">{{ __('Free') }}</button>
                <button type="button" class="btn btn-default" data-filter="paid">{{ __('Paid') }}</button>
                <button type="button" class="btn btn-default" data-filter="installed">{{ __('Installed') }}</button>
                <button type="button" class="btn btn-default" data-filter="not-installed">{{ __('Not Installed') }}</button>
            </div>

            <div class="table-responsive">
                <table class="table table-striped" id="modules-table" data-per-page="{{ $perPage }}">
                    <thead>
                        <tr>
                            <th>{{ __('Module') }}</th>
                            <th>{{ __('Description') }}</th>
                            <th>{{ __('Status') }}</th>
                            <th>{{ __('Action') }}</th>
                        </tr>
                    </thead>
                    <tbody>
                        @foreach ($modules as $module)
                            @php
                                $moduleSlug = $module['slug'] ?? '';
                                $isFree = !empty($module['is_free']);
                                $moduleIsInstalled = $isInstalled($moduleSlug);
                                $moduleIsEnabled = $isEnabled($moduleSlug);
                            @endphp
                            <tr class="module-row"
                                data-name="{{ strtolower($module['name'] ?? $module['slug']) }}"
                                data-description="{{ strtolower($module['description'] ?? '') }}"
                                data-slug="{{ strtolower($moduleSlug) }}"
                                data-free="{{ $isFree ? '1' : '0' }}"
                                data-installed="{{ ($moduleIsInstalled || $moduleIsEnabled) ? '1' : '0' }}">
                                <td>
                                    <strong>{{ $module['name'] ?? $module['slug'] }}</strong>
                                </td>
                                <td>{{ $module['description'] ?? '' }}</td>
                                <td>
                                    @if ($isFree)
                                        <span class="label label-success">{{ __('Free') }}</span>
                                    @else
                                        <span class="label label-warning">{{ __('Paid') }}</span>
                                    @endif
                                </td>
                                <td>
                                    @if ($moduleIsEnabled)
                                        <span class="text-muted">{{ __('Enabled') }}</span>
                                    @elseif ($moduleIsInstalled)
                                        <span class="text-muted">{{ __('Installed') }}</span>
                                    @else
                                        <form method="post" action="{{ route('freescoutinstaller_install') }}">
                                            {{ csrf_field() }}
                                            <input type="hidden" name="module_slug" value="{{ $moduleSlug }}">
                                            <button type="submit" class="btn btn-sm btn-primary">
                                                {{ __('Install') }}
                                            </button>
                                        </form>
                                    @endif
                                </td>
                            </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>

            <!-- No results message -->
            <div id="no-results" class="text-center text-muted" style="display: none; padding: 20px;">
                {{ __('No modules found matching your search.') }}
            </div>

            <!-- Pagination -->
            <div class="text-center" id="pagination-container">
                <ul class="pagination" id="pagination"></ul>
            </div>

            <script src="{{ route('freescoutinstaller_js') }}"></script>
        @endif
    </div>
</div>
