<?php

namespace Modules\FreescoutInstaller\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\File;
use Modules\FreescoutInstaller\Services\CatalogService;

class FreescoutInstallerController extends Controller
{
    public function settings()
    {
        $catalog = CatalogService::fetch();

        return view('freescoutinstaller::settings.index', [
            'modules' => $catalog['modules'] ?? [],
            'error' => $catalog['error'] ?? null,
        ]);
    }

public function install(Request $request)
{
    if (!Auth::user() || !Auth::user()->isAdmin()) {
        abort(403);
    }

    $slug = (string) $request->input('module_slug');
    if (!$slug) {
        return redirect()->back()->withErrors([
            'module_slug' => __('Module slug is required.'),
        ]);
    }

    $domain = CatalogService::getDomain();
    $instanceId = CatalogService::getInstanceId();

    // 1) Fetch catalog FIRST so we know if module is free/paid and trial settings
    $catalog = CatalogService::fetch();
    if (!empty($catalog['error'])) {
        return redirect()->back()->withErrors([
            'module_slug' => $catalog['error'],
        ]);
    }

    $moduleRow = null;
    foreach (($catalog['modules'] ?? []) as $m) {
        if (($m['slug'] ?? '') === $slug) {
            $moduleRow = $m;
            break;
        }
    }

    if (!$moduleRow) {
        return redirect()->back()->withErrors([
            'module_slug' => __('Module not found in catalog.'),
        ]);
    }

    $isFree = !empty($moduleRow['is_free']);
    $trialAvailable = !empty($moduleRow['trial_available']);

    // SHA from catalog (can be overridden by response header)
    $expectedSha = $moduleRow['sha256'] ?? null;

    // 2) TRIAL LOGIC ONLY FOR PAID MODULES
if (!$isFree) {

    // If trial is not available, DON'T block install (module will enforce license later)
    if (!$trialAvailable) {
        session()->flash('warning', __('This module requires a license key. You can still install it, then activate later.'));
    } else {

        $trial = CatalogService::fetchTrialStatus($slug, $domain);
        $status = $trial['status'] ?? null;

        // If API returned an error or unknown status, do not block install
        if (!empty($trial['error'])) {
            session()->flash('warning', __('Trial status unavailable. You can still install the module and activate later with a license.'));
        } else {

            // If never started, try to start the trial but never block install
            if ($status === 'never_started') {
                $start = CatalogService::startTrial($slug, $domain, $instanceId);
                if (empty($start['ok'])) {
                    session()->flash('warning', __('Trial could not be started. You can still install the module and activate later with a license.'));
                }
            }
            // If expired/already_used, still allow install (module enforces lock later)
            elseif ($status === 'expired' || $status === 'already_used') {
                session()->flash('warning', __('Trial expired. Module can be installed but will remain locked until a license is provided.'));
            }
            // Any other non-active status: allow install with warning
            elseif ($status !== 'active') {
                session()->flash('warning', __('Trial status unavailable. Module can be installed but may remain locked until a license is provided.'));
            }
        }
    }
}
    // 3) Download ZIP from Worker (works for BOTH free and paid)
    $storageDir = storage_path('app/freescout-installer');
    if (!File::isDirectory($storageDir)) {
        File::makeDirectory($storageDir, 0775, true);
    }
    $zipPath = $storageDir.'/'.$slug.'.zip';

    try {
        $client = new Client();
        $response = $client->request('POST', 'https://freescout-installer-api.suivi.workers.dev/api/v1/download', [
            'json' => [
                'module_slug' => $slug,
                'domain' => $domain,
                'instance_id' => $instanceId,
            ],
            'timeout' => 30,
            'connect_timeout' => (int) config('app.curl_connect_timeout', 5),
            'proxy' => config('app.proxy'),
            'stream' => true,
        ]);
    } catch (GuzzleException $e) {
        Log::warning('FreescoutInstaller download failed: '.$e->getMessage());
        return redirect()->back()->withErrors([
            'module_slug' => __('Download failed. Please try again later.'),
        ]);
    }

    $httpStatus = $response->getStatusCode();
    if ($httpStatus < 200 || $httpStatus >= 300) {
        return redirect()->back()->withErrors([
            'module_slug' => __('Download failed (HTTP :status).', ['status' => $httpStatus]),
        ]);
    }

    $out = @fopen($zipPath, 'wb');
    if (!$out) {
        return redirect()->back()->withErrors([
            'module_slug' => __('Could not write ZIP file to storage.'),
        ]);
    }

    $body = $response->getBody();
    while (!$body->eof()) {
        fwrite($out, $body->read(1024 * 1024));
    }
    fclose($out);

    // 4) Verify SHA256 (header overrides catalog if present)
    $headerSha = $response->getHeaderLine('X-Module-Sha256');
    $expectedSha = $headerSha ?: $expectedSha;

    if (!$expectedSha) {
        return redirect()->back()->withErrors([
            'module_slug' => __('Could not verify module hash (missing SHA256).'),
        ]);
    }

    $actualSha = hash_file('sha256', $zipPath);
    if (strtolower($actualSha) !== strtolower($expectedSha)) {
        return redirect()->back()->withErrors([
            'module_slug' => __('Hash mismatch. Download may be corrupted.'),
        ]);
    }

    // 5) Extract ZIP to Modules/
    $modulesDir = base_path('Modules');
    $zip = new \ZipArchive();
    if ($zip->open($zipPath) !== true) {
        return redirect()->back()->withErrors([
            'module_slug' => __('Could not open ZIP archive.'),
        ]);
    }

    $topFolder = $this->getTopLevelFolderFromZip($zip);

    if (!$zip->extractTo($modulesDir)) {
        $zip->close();
        return redirect()->back()->withErrors([
            'module_slug' => __('Could not extract ZIP archive.'),
        ]);
    }
    $zip->close();

    $this->clearModuleCache();
    $this->fixPermissions($modulesDir);
    $this->clearCaches();

    Log::info('FreescoutInstaller installed module: '.$slug);

    return redirect()->back()
        ->with('success', __('Installed module: :slug', ['slug' => $slug]))
        ->with('warning', __('Module installed. Please enable it from the Modules page.'));
}

    protected function fixPermissions($path)
    {
        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($path, \FilesystemIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $item) {
            @chmod($item->getPathname(), $item->isDir() ? 0775 : 0664);
            if (function_exists('posix_getuid')) {
                @chown($item->getPathname(), 'www-data');
                @chgrp($item->getPathname(), 'www-data');
            }
        }
    }

    protected function clearCaches()
    {
        try {
            Artisan::call('cache:clear');
            Artisan::call('view:clear');
            Artisan::call('config:clear');
            Artisan::call('route:clear');
        } catch (\Throwable $e) {
            Log::warning('FreescoutInstaller cache clear failed: '.$e->getMessage());
        }

        try {
            $cacheDirs = [
                storage_path('framework/cache'),
                base_path('bootstrap/cache'),
            ];
            foreach ($cacheDirs as $dir) {
                if (File::isDirectory($dir)) {
                    File::delete(File::allFiles($dir));
                }
            }
        } catch (\Throwable $e) {
            Log::warning('FreescoutInstaller cache file cleanup failed: '.$e->getMessage());
        }
    }

    protected function clearModuleCache()
    {
        try {
            $paths = array_merge(
                File::glob(base_path('bootstrap/cache/*modules*')) ?: [],
                File::glob(base_path('bootstrap/cache/*module*')) ?: []
            );
            foreach ($paths as $file) {
                if (File::isFile($file)) {
                    File::delete($file);
                }
            }
        } catch (\Throwable $e) {
            Log::warning('FreescoutInstaller module cache clear failed: '.$e->getMessage());
        }
    }

    protected function getTopLevelFolderFromZip(\ZipArchive $zip)
    {
        $top = null;
        for ($i = 0; $i < $zip->numFiles; $i++) {
            $name = $zip->getNameIndex($i);
            if (!$name) {
                continue;
            }
            $parts = explode('/', $name);
            if (!empty($parts[0])) {
                $top = $parts[0];
                break;
            }
        }
        return $top;
    }

    protected function readModuleInfo($modulesDir, $topFolder)
    {
        $info = [
            'name' => null,
            'alias' => null,
        ];

        if (!$topFolder) {
            return $info;
        }

        $moduleJson = $modulesDir.'/'.$topFolder.'/module.json';
        if (!File::exists($moduleJson)) {
            return $info;
        }

        $data = json_decode(File::get($moduleJson), true);
        if (!is_array($data)) {
            return $info;
        }

        $info['name'] = $data['name'] ?? null;
        $info['alias'] = $data['alias'] ?? null;

        return $info;
    }
}
